#include "MTApplication.h"
#include "IErrors.h"
#include "MTUtilities.h"
#include "MTUpdateObject.h"
#include "MTFileList.h"
#include "MTExceptions.h"
#include "AppearanceHelpers.h"
#include "MTSimpleResource.h"

#if DEBUG_LOG
std::ofstream	debugLog("log.txt");
#endif

//	MTApplication::MTApplication()
//	
//	application constructor
MTApplication::MTApplication()
{
	isDone = 0;
}

//	MTApplication::Init(void)
//	
//	init function
void MTApplication::Init(void)
{
	InitToolbox();
	
	if(hasNavigation)	// preload navigation services if it's installed
		NavLoad();
	
	if(hasAGL)	// start up OpenGL
		aglContext.Create(300, 300);
	
	menuBar.InstallMBAR(128);
	UpdateMenuState(kCompleteMenuUpdate);
	
	modalFilterProcPtr = NewModalFilterUPP(ModalFilterProc);
	if(!modalFilterProcPtr)
		throw MTException("Error creating ModalFilterUPP");
	
	InitAppleEventHandlers();
}

//	MTApplication::~MTApplication()
//	
//	application destructor, deinits libraries and toolbox
MTApplication::~MTApplication()
{
	DeInitAppleEventHandlers();
	
	if(modalFilterProcPtr)
	{
		DisposeModalFilterUPP(modalFilterProcPtr);
		modalFilterProcPtr = nil;
	}
	
	FreeAllWindows();
	
	if(hasAGL)
		aglContext.Dispose();
	
	if(hasNavigation)
		NavUnload();
	
	UnregisterAppearanceClient();
}

//	void MTApplication::Run(void)
//	
//	runs the application
void MTApplication::Run(void)
{
	while(!isDone)
	{
		HandleEvent();
	}
}

//	void MTApplication::ObjectUpdated(UInt32 type, UInt32 object)
//	
//	sends an object updated message
//	
//	type = object type
//	object = object id
void MTApplication::ObjectUpdated(UInt32 type, UInt32 object)
{
	for(UInt32 i = 0; i < windowList.size(); i++)
	{
		MTUpdateObject	* theObject = dynamic_cast<MTUpdateObject *>(windowList[i]);
		
		if(theObject)
			theObject->ObjectUpdated(type, object);
	}
}

//	void MTApplication::ObjectDeleted(UInt32 type, UInt32 object)
//	
//	sends an object deleted message
//	
//	type = object type
//	object = object id
void MTApplication::ObjectDeleted(UInt32 type, UInt32 object)
{
	for(UInt32 i = 0; i < windowList.size(); i++)
	{
		MTUpdateObject	* theObject = dynamic_cast<MTUpdateObject *>(windowList[i]);
		
		if(theObject)
			theObject->ObjectDeleted(type, object);
	}
}

//	void MTApplication::CloseAllOwnedBy(MTWindow * theWindow)
//	
//	closes all windows owned by a specified window
//	
//	theWindow = parent window
void MTApplication::CloseAllOwnedBy(MTWindow * theWindow)
{
	retry:
	
	for(SInt32 i = 0; i < windowList.size(); i++)
	{
		if(windowList[i]->OwnedBy(theWindow))
		{
			MTWindow	* theWindow = windowList[i];
			
			windowList.erase(windowList.begin() + i);
			
			theWindow->Close();
			
			delete theWindow;
			
			goto retry;	// loop back, deletion may have changed window list
		}
	}
	
	UpdateMenuState(kChangedWindowList);
}

//	void MTApplication::SendMessageToTopWindow(UInt32 type, UInt32 data)
//	
//	sends a message to the top window
//	
//	type = message type
//	data = message data
void MTApplication::SendMessageToTopWindow(UInt32 type, UInt32 data)
{
	MTWindow * theWindow = GetMTWindowFromWindowPtr(FrontWindow());
	
	if(theWindow)
		theWindow->RecieveMessage(type, data);
}

UInt8 MTApplication::TopWindowSupportsMessage(UInt32 type, UInt32 data)
{
	MTWindow * theWindow = GetMTWindowFromWindowPtr(FrontWindow());
	
	if(theWindow)
		return theWindow->SupportsMessage(type, data);
	
	return 0;
}

//	void MTApplication::DoMoveableModalDialog(short id, short defaultItem, short cancelItem, ModalDialogEventHandler eventHandler, long refCon)
//	
//	handles a moveable modal dialog box
//	
//	id = resource id
//	defaultItem = index of default item
//	cancleItem = index of cancelItem
//	eventHandler = pointer to event handling function
//	refCon = dialog reference constant
void MTApplication::DoMoveableModalDialog(short id, short defaultItem, short cancelItem, ModalDialogEventHandler eventHandler, long refCon)
{
	DialogPtr	theDialog;
	short		item;
	
	theDialog = GetNewDialog(id, nil, (WindowPtr)-1);
	
	SetWRefCon(GetDialogWindow(theDialog), refCon);
	
	if(defaultItem)
		SetDialogDefaultItem(theDialog, defaultItem);
	if(cancelItem)
		SetDialogCancelItem(theDialog, cancelItem);
	
	SetDialogTracksCursor(theDialog, true);
	
	if(eventHandler)
	{
		UInt8	keepGoing;
		
		do
		{
			ModalDialog(modalFilterProcPtr, &item);
			
			keepGoing = eventHandler(theDialog, item);
		}
		while(keepGoing == 1);
	}
	else
	{
		do
		{
			ModalDialog(modalFilterProcPtr, &item);
		}
		while(item != 1);
	}
	
	DisposeDialog(theDialog);
}

//	void MTApplication::InitToolbox(void)
//	
//	inits the toolbox, seeds the random generator, and sets up the appearance manager
void MTApplication::InitToolbox(void)
{
	InitGraf(&qd.thePort);
	InitFonts();
	InitWindows();
	InitMenus();
	TEInit();
	InitDialogs(nil);
	InitCursor();
	
	for(UInt32 i = 0; i < 10; i++)
		MoreMasters();
	
	MaxApplZone();
	
	GetDateTime((UInt32 *)&qd.randSeed);	// seed random generator
	
	RegisterAppearanceClient();	// inits appearance
	
	hasNavigation = ((void *)NavLoad != (void *)kUnresolvedCFragSymbolAddress);
	hasAGL =		((void *)aglCreateContext != (void *)kUnresolvedCFragSymbolAddress);
}

void MTApplication::InitAppleEventHandlers(void)
{
	OSErr	theErr;
	
	openApplicationHandlerUPP =	NewAEEventHandlerUPP(HandleOpenApplicationEvent);
	openDocumentHandlerUPP =	NewAEEventHandlerUPP(HandleOpenDocumentsEvent);
	printDocumentHandlerUPP =	NewAEEventHandlerUPP(HandlePrintDocumentsEvent);
	quitApplicationHandlerUPP =	NewAEEventHandlerUPP(HandleQuitApplicationEvent);
	
	if(	!openApplicationHandlerUPP ||
		!openDocumentHandlerUPP ||
		!printDocumentHandlerUPP ||
		!quitApplicationHandlerUPP)
		throw MTException("Error creating Apple Event UPPs");
	
	theErr = AEInstallEventHandler(kCoreEventClass, kAEOpenApplication, openApplicationHandlerUPP, 0, 0);
	if(theErr)
		throw MTException("Error installing AppleEvent handler procedure");
	
	theErr = AEInstallEventHandler(kCoreEventClass, kAEOpenDocuments, openDocumentHandlerUPP, 0, 0);
	if(theErr)
		throw MTException("Error installing AppleEvent handler procedure");
	
	theErr = AEInstallEventHandler(kCoreEventClass, kAEPrintDocuments, printDocumentHandlerUPP, 0, 0);
	if(theErr)
		throw MTException("Error installing AppleEvent handler procedure");
	
	theErr = AEInstallEventHandler(kCoreEventClass, kAEQuitApplication, quitApplicationHandlerUPP, 0, 0);
	if(theErr)
		throw MTException("Error installing AppleEvent handler procedure");
}

void MTApplication::DeInitAppleEventHandlers(void)
{
	AERemoveEventHandler(kCoreEventClass, kAEOpenApplication, openApplicationHandlerUPP, 0);
	AERemoveEventHandler(kCoreEventClass, kAEOpenDocuments, openDocumentHandlerUPP, 0);
	AERemoveEventHandler(kCoreEventClass, kAEPrintDocuments, printDocumentHandlerUPP, 0);
	AERemoveEventHandler(kCoreEventClass, kAEQuitApplication, quitApplicationHandlerUPP, 0);
	
	DisposeAEEventHandlerUPP(openApplicationHandlerUPP);
	DisposeAEEventHandlerUPP(openDocumentHandlerUPP);
	DisposeAEEventHandlerUPP(printDocumentHandlerUPP);
	DisposeAEEventHandlerUPP(quitApplicationHandlerUPP);
}

//	void MTApplication::HandleEvent(void)
//	
//	handles one event
void MTApplication::HandleEvent(void)
{
	EventRecord	theEvent;
	
	WaitNextEvent(everyEvent, &theEvent, 2, nil);
	
	if(FrontWindow())
		IdleControls(FrontWindow());
	
	switch(theEvent.what)
	{
		case mouseDown:
		{
			WindowPtr	targetWindow;
			short		where;
			
			where = FindWindow(theEvent.where, &targetWindow);
			
			switch(where)
			{
				case inMenuBar:
					HandleMenuSelection(MenuSelect(theEvent.where));
					break;
				
				case inContent:
					if(targetWindow != FrontWindow())
					{
						MTWindow *		theWindow = GetMTWindowFromWindowPtr(targetWindow);
						
						if(theWindow)
						{
							theWindow->BringToFront();
						}
						else
						{
							SelectWindow(targetWindow);
						}
						
						UpdateMenuState(kChangedWindowOrder);
					}
					else
					{
						MTWindow *		theWindow = GetMTWindowFromWindowPtr(targetWindow);
						static Point	lastDownLoc = {0, 0};
						static long		lastDownTime = 0;
						UInt32			flags = 0;
						
						if(	(std::abs(lastDownLoc.h - theEvent.where.h) < 5) &&	// double click is two clicks within
							(std::abs(lastDownLoc.v - theEvent.where.v) < 5) &&	// GetDblTime() ticks and without moving
							((theEvent.when - lastDownTime) < GetDblTime())	)	// more than 5 pixels in either direction
						{
							flags |= kIsDoubleClick;
						}
						
						if(theWindow)
						{
							theWindow->HandleContentEvent(&theEvent, flags);
						}
						
						lastDownLoc = theEvent.where;	// update old positions
						lastDownTime = theEvent.when;
					}
					break;
				
				case inDrag:
					DragWindow(targetWindow, theEvent.where, &qd.screenBits.bounds);
					break;
				
				case inGrow:
					if(targetWindow == FrontWindow())
					{
						MTWindow *	theWindow = GetMTWindowFromWindowPtr(targetWindow);
						Rect		bounds;
						
						if(theWindow)
							theWindow->GetZoomSizeBounds(&bounds);
						else
							bounds = qd.screenBits.bounds;
						
						long	newSize = GrowWindow(targetWindow, theEvent.where, &bounds);
						
						if(newSize != 0)
						{
							SizeWindow(targetWindow, LoWord(newSize), HiWord(newSize), 0);
							
							if(theWindow)
								theWindow->HandleResizeEvent(LoWord(newSize), HiWord(newSize));
							
							Rect	junk;
							InvalWindowRect(targetWindow, GetWindowPortBounds(targetWindow, &junk));
						}
					}
					else
					{
						SelectWindow(targetWindow);
					}
					break;
				
				case inGoAway:
					if(TrackGoAway(targetWindow, theEvent.where))
					{
						FreeWindow(targetWindow);
					}
					break;
				
				case inZoomIn:
				case inZoomOut:
					if(TrackBox(targetWindow, theEvent.where, where))
					{
						MTGrafPortSaver	saver(targetWindow);
						
						ZoomWindow(targetWindow, where, 1);
						
						Rect	bounds;
						GetWindowBounds(targetWindow, kWindowContentRgn, &bounds);
						
						MTWindow *	theWindow = GetMTWindowFromWindowPtr(targetWindow);
						if(theWindow)
							theWindow->HandleResizeEvent(bounds.right - bounds.left, bounds.bottom - bounds.top);
					}
					break;
			}
		}
		break;
		
		case keyDown:
		{
			long	selection = MenuEvent(&theEvent);
			
			if(selection == 0)
			{
				MTWindow *	theWindow = GetMTWindowFromWindowPtr(FrontWindow());
				
				if(theWindow)
					theWindow->HandleKeyEvent(&theEvent, theEvent.message & charCodeMask, (theEvent.message >> 8) & keyCodeMask, theEvent.modifiers);
			}
			else
			{
				HandleMenuSelection(selection);
			}
		}
		break;
		
		case autoKey:
		{
			MTWindow *	theWindow = GetMTWindowFromWindowPtr(FrontWindow());
			
			if(theWindow)
				theWindow->HandleKeyEvent(&theEvent, theEvent.message & charCodeMask, (theEvent.message >> 8) & keyCodeMask, theEvent.modifiers);
		}
		break;
		
		case updateEvt:
		{
			HandleUpdateEvent(&theEvent);
		}
		break;
		
		case kHighLevelEvent:
		{
			AEProcessAppleEvent(&theEvent);
		}
		break;
	}
}

void MTApplication::HandleUpdateEvent(EventRecord * theEvent)
{
	MTWindow *		theWindow;
	MTGrafPortSaver	saver(theEvent->message);
	
	theWindow = GetMTWindowFromWindowPtr((WindowPtr)theEvent->message);
	
	BeginUpdate((WindowPtr)theEvent->message);
	
	DrawControls((WindowPtr)theEvent->message);
	
	if(theWindow)
		theWindow->HandleUpdateEvent();
	
	EndUpdate((WindowPtr)theEvent->message);
}

//	void MTApplication::HandleMenuSelection(long selection)
//	
//	dispatches a menu selection to the correct handler
//	
//	selection = menu selection	(low word = menu item, high word = menu id)
void MTApplication::HandleMenuSelection(long selection)
{
	short	item = LoWord(selection);
	long	ticks = TickCount();
	
	try
	{
		switch(HiWord(selection))
		{
			case kAppleMenu:
				switch(item)
				{
					case kAppleAboutItem:
						DoMoveableModalDialog(1000, 1, 0, nil, nil);
						break;
					
					default:
					{
						Str255	itemName;
						GetMenuItemText(menuBar.MyGetMenuHandle(kAppleMenu), item, itemName);
						
						OpenDeskAcc(itemName);
					}
					break;
				}
				break;
			
			case kFileMenu:
				switch(item)
				{
					case kFileOpenItem:
						DoOpenFile();
						break;
					
					case kFileCloseItem:
						FreeWindow(FrontWindow());
						break;
					
					case kFileCloseAllItem:
						FreeAllWindows();
						break;
					
					case kFileGetInfoItem:
						SendMessageToTopWindow('INFO', 0);
						break;
					
					case kFileImportItem:
						SendMessageToTopWindow('IMPT', 0);
						break;
					
					case kFileExportItem:
						SendMessageToTopWindow('EXPT', 0);
						break;
					
					case kFileApplyCMPOPatchItem:
						ApplyCMPO();
						break;
					
					case kFileRemoveCMPOPatchItem:
						RemoveCMPO();
						break;
					
					case kFileCreateCMPOPatchItem:
						CreateCMPO();
						break;
					
					case kFileQuitItem:
						isDone = 1;
						break;
				}
				break;
			
			case kEditMenu:
				switch(item)
				{
					case kEditUndoItem:
						break;
					
					case kEditCutItem:
						break;
					
					case kEditCopyItem:
						break;
					
					case kEditPasteItem:
						break;
					
					case kEditClearItem:
						break;
					
					case kEditSelectAllItem:
						break;
					
					case kEditPreferencesItem:
						break;
				}
				break;
			
			case kResourceMenu:
				switch(item)
				{
					case kResourceOpenByIDItem:
						DoOpenResourceByID();
						break;
					
					case kResourceEditItem:
						SendMessageToTopWindow('EDIT', 0);
						break;
					
					case kResourceTextDumpItem:
						SendMessageToTopWindow('TDMP', 0);
						break;
				}
				break;
			
			case kWindowsMenu:
				if((item - 1) < windowList.size())
				{
					windowList[item - 1]->BringToFront();
					
					UpdateMenuState(kChangedWindowOrder);
				}
				
				break;
		}
	}
	catch(MTException theException)
	{
		theException.DoErrorDialog();
	}
	
	while(ticks == TickCount()) ;	// leave it hilited during this tick
	
	HiliteMenu(0);
}

//	void MTApplication::UpdateMenuState(UInt32 needsUpdate)
//	
//	updates the current state of the menus to reflect the application's state
//	
//	needsUpdate = bitmask containing what to update
void MTApplication::UpdateMenuState(UInt32 needsUpdate)
{
	if(needsUpdate & kChangedWindowList)
	{
		if(windowList.size())
		{
			menuBar.EnableItem(kFileMenu, kFileCloseItem);
			menuBar.EnableItem(kFileMenu, kFileCloseAllItem);
			menuBar.EnableItem(kFileMenu, kFileGetInfoItem);
			menuBar.EnableItem(kFileMenu, kFileExportItem);
			menuBar.EnableItem(kFileMenu, kFileImportItem);
			menuBar.EnableMenu(kResourceMenu);
			menuBar.EnableMenu(kWindowsMenu);
		}
		else
		{
			menuBar.DisableItem(kFileMenu, kFileCloseItem);
			menuBar.DisableItem(kFileMenu, kFileCloseAllItem);
			menuBar.DisableItem(kFileMenu, kFileGetInfoItem);
			menuBar.DisableItem(kFileMenu, kFileExportItem);
			menuBar.DisableItem(kFileMenu, kFileImportItem);
			menuBar.DisableMenu(kResourceMenu);
			menuBar.DisableMenu(kWindowsMenu);
		}
		
		{
			MenuHandle	windowMenu = menuBar.MyGetMenuHandle(kWindowsMenu);
			short		numItems = CountMItems(windowMenu);
			
			for(UInt32 i = 0; i < numItems; i++)	// empty the menu
				DeleteMenuItem(windowMenu, 1);
			
			for(UInt32 i = 0; i < windowList.size(); i++)
			{
				Str255	theString;
				
				windowList[i]->GetWindowName(theString);
				
				MacInsertMenuItem(windowMenu, theString, 32767);
				
				if(i <= 9)
					SetItemCmd(windowMenu, i + 1, '0' + i);
				else
					SetItemCmd(windowMenu, i + 1, 0);
			}
		}
	}
	
	if(	(needsUpdate & kChangedWindowList) ||
		(needsUpdate & kChangedWindowOrder))
	{
		MenuHandle	windowMenu = menuBar.MyGetMenuHandle(kWindowsMenu);
		
		for(UInt32 i = 0; i < windowList.size(); i++)
		{
			if(*windowList[i] == FrontWindow())
			{
				SetItemMark(windowMenu, i + 1, checkMark);
			}
			else
			{
				SetItemMark(windowMenu, i + 1, noMark);
			}
		}
		
		if(dynamic_cast<MTSimpleResourceInfoWindow *>(GetMTWindowFromWindowPtr(FrontWindow())))
		{
			menuBar.EnableItem(kResourceMenu, kResourceEditItem);
		}
		else
		{
			menuBar.DisableItem(kResourceMenu, kResourceEditItem);
		}
	}
	
	if(	(needsUpdate & kChangedMessageSupport) ||
		(needsUpdate & kChangedWindowOrder) ||
		(needsUpdate & kChangedWindowList))
	{
		if(FrontWindow())
		{
			if(TopWindowSupportsMessage('INFO', 0))
				menuBar.EnableItem(kFileMenu, kFileGetInfoItem);
			else
				menuBar.DisableItem(kFileMenu, kFileGetInfoItem);
			
			if(TopWindowSupportsMessage('IMPT', 0))
				menuBar.EnableItem(kFileMenu, kFileImportItem);
			else
				menuBar.DisableItem(kFileMenu, kFileImportItem);
			
			if(TopWindowSupportsMessage('EXPT', 0))
				menuBar.EnableItem(kFileMenu, kFileExportItem);
			else
				menuBar.DisableItem(kFileMenu, kFileExportItem);
			
			if(TopWindowSupportsMessage('EDIT', 0))
				menuBar.EnableItem(kResourceMenu, kResourceEditItem);
			else
				menuBar.DisableItem(kResourceMenu, kResourceEditItem);
			
			if(TopWindowSupportsMessage('TDMP', 0))
				menuBar.EnableItem(kResourceMenu, kResourceTextDumpItem);
			else
				menuBar.DisableItem(kResourceMenu, kResourceTextDumpItem);
		}
	}
	
	menuBar.UpdateMenuBar();
}

//	MTWindow * MTApplication::LookupWindow(WindowPtr theWindow, UInt32 * id)
//	
//	given a WindowPtr, returns an MTWindow and an index into the window list
//	
//	theWindow = system window pointer
//	id = pointer to index (can be nil, set to 0xFFFFFFFF if window not found)
//	returns pointer to MTWindow
MTWindow * MTApplication::LookupWindow(WindowPtr theWindow, UInt32 * id)
{
	for(UInt32 i = 0; i < windowList.size(); i++)
	{
		if(*windowList[i] == theWindow)
		{
			if(id)
				*id = i;
			return windowList[i];
		}
	}
	
	if(id)
		*id = 0xFFFFFFFF;	// invalid value
	
	return nil;
}

//	MTWindow * MTWindow::AttemptCreateWindowOfType(MTWindowType type)
//	
//	attempts to create and add a window of a specified type
//	
//	type = type of window to create
//	returns new window
MTWindow * MTApplication::AttemptCreateWindowOfType(MTWindowType type)
{
	MTWindow	* theWindow;
	
	if(HasWindowOfType(type, &theWindow))
	{
		ASSERT(theWindow, "MTWindow::AttemptCreateWindowOfType: found window type, but got nil pointer");
		
		if(theWindow->IsUniqueWindow())	// bring to front if it's unique
		{
			theWindow->BringToFront();
			
			UpdateMenuState(kChangedWindowOrder);
			
			return nil;
		}
	}
	
	theWindow = CreateWindowOfType(type);
	
	AttachWindow(theWindow);
	
	return theWindow;
}

//	void MTApplication::AttachWindow(MTWindow * theWindow)
//	
//	adds an MTWindow to the window list
//	
//	theWindow = pointer to window to add
void MTApplication::AttachWindow(MTWindow * theWindow)
{
	if(theWindow)
	{
		windowList.push_back(theWindow);
		
		theWindow->Open();
		
		UpdateMenuState(kChangedWindowList);
	}
}

//	void MTApplication::FreeWindow(WindowPtr targetWindow)
//	
//	closes and removes a window from the window list
//	
//	targetWindow = pointer to window to remove
void MTApplication::FreeWindow(WindowPtr targetWindow)
{
	if(targetWindow)
	{
		UInt32		id;
		MTWindow *	theWindow = LookupWindow(targetWindow, &id);
		
		if(theWindow)
		{
			theWindow->Close();
			
			delete theWindow;
			
			windowList.erase(windowList.begin() + id);
		}
		else
			DisposeWindow(targetWindow);
		
		UpdateMenuState(kChangedWindowList);
	}
}

//	void MTApplication::FreeAllWindows(void)
//	
//	closes all windows and removes them from the window list
void MTApplication::FreeAllWindows(void)
{
	while(!windowList.empty())
	{
		MTWindow	* theWindow = windowList[0];
		
		windowList.erase(windowList.begin());
		
		theWindow->Close();
		
		delete theWindow;
	}
	
	UpdateMenuState(kChangedWindowList);
}

//	UInt8 MTApplication::HasWindowOfType(MTWindowType type)
//	
//	returns if the window list contains a window of a type
//	
//	type = window type constant
//	returns if such window exists
UInt8 MTApplication::HasWindowOfType(MTWindowType type)
{
	for(UInt32 i = 0; i < windowList.size(); i++)
	{
		if(windowList[i]->GetBaseWindowType() == type)
		{
			return 1;
		}
	}
	
	return 0;
}

//	UInt8 MTApplication::HasWindowOfType(MTWindowType type, MTWindow ** theWindow)
//	
//	returns if the window list contains a window of a type
//	
//	type = window type constant
//	returns if such window exists
UInt8 MTApplication::HasWindowOfType(MTWindowType type, MTWindow ** theWindow)
{
	for(UInt32 i = 0; i < windowList.size(); i++)
	{
		if(windowList[i]->GetBaseWindowType() == type)
		{
			*theWindow = windowList[i];
			
			return 1;
		}
	}
	
	return 0;
}

//	MTWindow * MTApplication::CreateWindowOfType(MTWindowType type)
//	
//	returns a new window of a specified type
//	
//	type = type of window to create
MTWindow * MTApplication::CreateWindowOfType(MTWindowType type)
{
	switch(type)
	{
		
	}
	
	return nil;
}

//	void MTApplication::DoOpenFile(void)
//	
//	handles opening a new file
void MTApplication::DoOpenFile(void)
{
	OSErr				theErr;
	NavReplyRecord		reply;
	NavDialogOptions	options;
	NavEventUPP			eventHandler;
	
	theErr = NavGetDefaultDialogOptions(&options);
	if(theErr)
		throw MTOSException(theErr, "Error getting default Navigation options");
	
	options.dialogOptionFlags &= ~kNavAllowPreviews;
	pstrcpy(options.clientName, "\pOniTools");
	pstrcpy(options.message, "\pPlease select a .dat file.");
	pstrcpy(options.windowTitle, "\pOpen File");
	
	eventHandler = NewNavEventUPP(NavigationEventProc);
	theErr = NavGetFile(nil, &reply, &options, eventHandler, nil, nil, nil, (void *)this);
	DisposeNavEventUPP(eventHandler);
	
	if(theErr && (theErr != userCanceledErr))
		throw MTOSException(theErr, "Navigation Services Error");
	
	if(reply.validRecord)
	{
		DoOpenFile(&reply.selection);
		
		theErr = NavDisposeReply(&reply);
		if(theErr)
			throw MTOSException(theErr, "Navigation Services error");
	}
}

void MTApplication::DoOpenFile(AEDescList * descList)
{
	OSErr	theErr;
	long	itemCount;
	
	theErr = AECountItems(descList, &itemCount);
	if(theErr)
		throw MTOSException(theErr, "Error parsing AppleEvent");
	
	for(UInt32 i = 1; i <= itemCount; i++)
	{
		AEKeyword	theKeyword;
		DescType	actualType;
		Size		actualSize;
		FSSpec		theFile;
		
		theErr = AEGetNthPtr(descList, i, typeFSS, &theKeyword, &actualType, &theFile, sizeof(FSSpec), &actualSize);
		if(!theErr)
		{
			try
			{
				DoOpenFile(&theFile);
			}
			catch(MTException theException)
			{
				// handle exceptions here, continue loading files
				
				theException.DoErrorDialog();
			}
		}
	}
}

void MTApplication::DoOpenFile(FSSpec * theFileSpec)
{
	for(UInt32 i = 0; i < windowList.size(); i++)	// check if the file is already open
	{
		MTFileList	* theObject = windowList[i]->GetOwningFileList();
		
		if(theObject && CompareFSSpec(&theObject->theFile->baseFile, theFileSpec))
		{
			theObject->BringToFront();
			
			UpdateMenuState(kChangedWindowOrder);
			
			return;
		}
	}
	
	MTFile * theFile = new MTFile;
	
	try
	{
		theFile->LoadFile(theFileSpec);
	}
	catch(MTException err)
	{
		err.DoErrorDialog();
		
		delete theFile;
		
		return;
	}
	
	MTFileList * theWindow = new MTFileList;
	
	theWindow->AttachFile(theFile);
	
	AttachWindow(theWindow);
}

void MTApplication::DoOpenResourceByID(void)
{
	OpenResourceByIDRecord	record;
	
	DoMoveableModalDialog(1100, 1, 2, DoOpenResourceByIDItemProc, (long)&record);
	
	if(!record.canceled)
	{
		MTFileList	* frontFileList = GetMTWindowFromWindowPtr(FrontWindow())->GetOwningFileList();
		
		if(frontFileList)
		{
			UInt32	idx = frontFileList->theFile->LookupIdxFromID(record.id);
			
			if(idx != 0xFFFFFFFF)
			{
				frontFileList->OpenViewerWindow(idx);
			}
			else
			{
				SysBeep(1);
			}
		}
		else
		{
			SysBeep(1);
		}
	}
}

UInt8 MTApplication::DoOpenResourceByIDItemProc(DialogPtr theDialog, short item)
{
	UInt8					keepGoing = 1;
	OpenResourceByIDRecord	* data = (OpenResourceByIDRecord *)GetWRefCon(GetDialogWindow(theDialog));
	
	switch(item)
	{
		case 1:
		{
			Str255			string;
			ControlHandle	theControl;
			char			buf[256];
			char			* end;
			
			GetDialogItemAsControl(theDialog, 4, &theControl);
			GetEditTextText(theControl, string);
			
			CopyPascalStringToC(string, buf);
			
			data->id = std::strtoul(buf, &end, 16);
			
			if(end == &buf[std::strlen(buf)])	// if the entire string wasn't read
			{									// the input was bad
				keepGoing = 0;
				data->canceled = 0;
			}
			else
			{
				SysBeep(1);
			}
		}
		break;
		
		case 2:
			keepGoing = 0;
			
			data->canceled = 1;
			break;
	}
	
	return keepGoing;
}

pascal Boolean MTApplication::ModalFilterProc(DialogRef theDialog, EventRecord * theEvent, DialogItemIndex * itemHit)
{
	if(theEvent && theEvent->what == updateEvt)
	{
		gTheApp->HandleUpdateEvent(theEvent);
		
		return true;
	}
	
	return StdFilterProc(theDialog, theEvent, itemHit);
}

pascal OSErr MTApplication::HandleOpenApplicationEvent(const AppleEvent * theAppleEvent, AppleEvent * theReply, long refCon)
{
	#pragma unused (theAppleEvent)
	#pragma unused (theReply)
	#pragma unused (refCon)
	
	return noErr;
}

pascal OSErr MTApplication::HandleOpenDocumentsEvent(const AppleEvent * theAppleEvent, AppleEvent * theReply, long refCon)
{
	#pragma unused (theReply)
	#pragma unused (refCon)
	
	OSErr		theErr;
	AEDescList	descList;
	
	theErr = AEGetParamDesc(theAppleEvent, keyDirectObject, typeAEList, &descList);
	if(theErr)
	{
		DoOSErr("Error handling Apple Event", theErr);
		
		return noErr;
	}
	
	gTheApp->DoOpenFile(&descList);
	
	theErr = AEDisposeDesc(&descList);
	
	return noErr;
}

pascal OSErr MTApplication::HandlePrintDocumentsEvent(const AppleEvent * theAppleEvent, AppleEvent * theReply, long refCon)
{
	#pragma unused (theAppleEvent)
	#pragma unused (theReply)
	#pragma unused (refCon)
	
	DoError("OniTools doesn't support printing. (what would it print?)");
	
	return noErr;
}

pascal OSErr MTApplication::HandleQuitApplicationEvent(const AppleEvent * theAppleEvent, AppleEvent * theReply, long refCon)
{
	#pragma unused (theAppleEvent)
	#pragma unused (theReply)
	#pragma unused (refCon)
	
	gTheApp->isDone = 1;
	
	return noErr;
}

UInt32 MTApplication::ReadCMPO(CMPODataRecord * record)
{
	IFileStream			patch;
	char				buf[4096];
	FSSpec				theFile;
	
	if(NavigationGetFileCustom(&theFile, "\pPlease locate the patch file."))
	{
		patch.OpenFile(&theFile);
		
		try
		{
			patch.ReadUnknownString(buf, 4096);
			
			if(std::strcmp(buf, "CMPO0"))
				throw MTUnsupportedException("Unsupported CMPO Version");
			
			patch.ReadUnknownString(buf, 4096);
			record->targetFile = buf;
			
			patch.ReadUnknownString(buf, 4096);
			record->targetOS = buf;
			
			while(!patch.HitEOF())
			{
				patch.ReadUnknownString(buf, 4096);
				
				if(buf[0])
				{
					CMPODataEntry	entry;
					UInt32			originalData, patchData;
					
					if(std::sscanf(buf, "%ld %lo %lo", &entry.offset, &originalData, &patchData) != 3)
						throw MTException("Corrupt CMPO File");
					
					entry.originalData = originalData;
					entry.patchData = patchData;
					
					record->entries.push_back(entry);
				}
			}
		}
		catch(MTException exception)
		{
			exception.DoErrorDialog();
		}
		
		patch.CloseFile();
	}
	else
	{
		return 1;
	}
	
	return 0;
}

void MTApplication::PrintCMPO(std::ostream & stream, CMPODataRecord * record)
{
	stream << record->targetFile << "\n";
	stream << record->targetOS << "\n";
	
	for(UInt32 i = 0; i < record->entries.size(); i++)
	{
		stream	<< (void *)record->entries[i].offset << " "
				<< (UInt32)record->entries[i].originalData << " "
				<< (UInt32)record->entries[i].patchData << "\n";
	}
}

void MTApplication::ApplyCMPO(void)
{
	CMPODataRecord	record;
	IFileStream		target;
	Str255			theString;
	char			buf[256];
	FSSpec			theFile;
	
	if(ReadCMPO(&record))
		return;
	
	std::sprintf(buf, "Please locate \"%s\".", record.targetFile.c_str());
	CopyCStringToPascal(buf, theString);
	
	if(NavigationGetFileCustom(&theFile, theString))
	{
		try
		{
			target.OpenFile(&theFile);
			
			for(UInt32 i = 0; i < record.entries.size(); i++)
			{
				target.SetPosition(record.entries[i].offset);
				
				target.WriteChar(record.entries[i].patchData);
			}
		}
		catch(MTException theException)
		{
			theException.DoErrorDialog();
		}
		
		target.CloseFile();
	}
}

void MTApplication::RemoveCMPO(void)
{
	CMPODataRecord	record;
	IFileStream		target;
	Str255			theString;
	char			buf[256];
	FSSpec			theFile;
	
	if(ReadCMPO(&record))
		return;
	
	std::sprintf(buf, "Please locate \"%s\".", record.targetFile.c_str());
	CopyCStringToPascal(buf, theString);
	
	if(NavigationGetFileCustom(&theFile, theString))
	{
		try
		{
			target.OpenFile(&theFile);
			
			for(UInt32 i = 0; i < record.entries.size(); i++)
			{
				target.SetPosition(record.entries[i].offset);
				
				target.WriteChar(record.entries[i].originalData);
			}
		}
		catch(MTException theException)
		{
			theException.DoErrorDialog();
		}
		
		target.CloseFile();
	}
}

void MTApplication::CreateCMPO(void)
{
	IFileStream	originalFile, editedFile, outputFile;
	FSSpec		theOriginalFile, theEditedFile, theOutputFile;
	char		buf[4096];
	UInt32		offset = 0;
	UInt8		isReplacing;
	OSErr		theErr;
	SInt32		remaining = originalFile.GetLength();
	UInt8		* compareBufOriginal = nil,
				* compareBufNew = nil;
	UInt32		bufSize = 1024 * 32;
	
	try
	{
		if(!NavigationGetFileCustom(&theOriginalFile, "\pPlease locate the original file."))
			return;
		
		originalFile.OpenFile(&theOriginalFile);
		
		if(!NavigationGetFileCustom(&theEditedFile, "\pPlease locate the edited file."))
			return;
		
		editedFile.OpenFile(&theEditedFile);
		
		if(originalFile.GetLength() != editedFile.GetLength())
		{
			DoError("These files are different sizes. Patch could not be created.");
			
			originalFile.CloseFile();
			editedFile.CloseFile();
			
			return;
		}
		
		if(!NavigationPutFileCustom(&theOutputFile, "\pPlease select a location for the patch file.", "\ppatch.cmpo", &isReplacing))
			return;
		
		if(isReplacing)
		{
			theErr = FSpDelete(&theOutputFile);
			if(theErr)
				throw MTOSFileException(theErr, "Error deleting old file");
		}
		
		outputFile.CreateFile(&theOutputFile, 'ONIt', 'TEXT');
		
		outputFile.WritePCString("CMPO0");
		
		CopyPascalStringToC(theOriginalFile.name, buf);
		outputFile.WritePCString(buf);
		
		outputFile.WritePCString("mac");
		
		remaining = originalFile.GetLength();
		
		compareBufOriginal = (UInt8 *)NewPtr(bufSize);
		if(!compareBufOriginal)
			throw MTMemoryException("Out of memory");
		
		compareBufNew = (UInt8 *)NewPtr(bufSize);
		if(!compareBufNew)
			throw MTMemoryException("Out of memory");
		
		while(remaining > 0)
		{
			UInt32	copySize = remaining;
			
			if(copySize > bufSize)
				copySize = bufSize;
			
			originalFile.ReadBuffer(compareBufOriginal, copySize);
			editedFile.ReadBuffer(compareBufNew, copySize);
			
			for(UInt32 i = 0; i < copySize; i++)
			{
				UInt32	originalData;
				UInt32	editedData;
				
				originalData = compareBufOriginal[i];
				editedData = compareBufNew[i];
				
				if(originalData != editedData)
				{
					std::sprintf(buf, "%d %o %o", offset, originalData, editedData);
					
					outputFile.WritePCString(buf);
				}
				
				offset++;
			}
			
			remaining -= copySize;
		}
	}
	catch(MTException theException)
	{
		theException.DoErrorDialog();
	}
	
	if(compareBufOriginal)
		DisposePtr((Ptr)compareBufOriginal);
	if(compareBufNew)
		DisposePtr((Ptr)compareBufNew);
	
	originalFile.CloseFile();
	editedFile.CloseFile();
	outputFile.CloseFile();
}